<?php
/**
 * Plugin Name: GTI Public Preview
 * Description: ログインなし、IP制限なし、特定のパラメータが付与されたURLで下書きプレビューを許可するプラグイン。
 * Version: 1.1.0
 * Author: ウェブ屋のさとーさん@GTI Inc. 4月から復帰
 * Author URI: https://blog.gti.jp
 * Text Domain: gti-public-preview
 */

if (!defined('ABSPATH')) exit;

if (!class_exists('GTI_Public_Preview')) {

    class GTI_Public_Preview
    {
        const OPTION_PARAM_KEY          = 'spp_param_key';
        const OPTION_ALLOWED_POST_TYPES = 'spp_allowed_post_types';
        const META_KEY_TOKEN            = '_spp_preview_token';
        const META_KEY_EXPIRATION       = '_spp_preview_expiration';
        const META_KEY_IS_ACTIVE        = '_spp_preview_is_active';

        public function __construct()
        {
            // 言語ファイルの読み込み
            add_action('plugins_loaded', [$this, 'load_textdomain']);

            // 管理画面メニュー
            add_action('admin_menu', [$this, 'add_admin_menu']);
            
            // 投稿保存時の処理（メタボックス保存）
            add_action('save_post', [$this, 'save_meta_box_data']);

            // メタボックス追加
            add_action('add_meta_boxes', [$this, 'add_custom_meta_box']);

            // 下書きプレビュー許可ロジック
            add_filter('the_posts', [$this, 'allow_draft_preview'], 10, 2);
            
            // プレビューリンクに自動でパラメータを付与
            add_filter('preview_post_link', [$this, 'filter_preview_link'], 10, 2);
            
            // 管理バーに明示的なリンクを表示
            add_action('admin_bar_menu', [$this, 'add_preview_link_to_admin_bar'], 80);

            // noindex の出力
            add_action('wp_head', [$this, 'add_noindex_meta'], 1);
            add_filter('wp_headers', [$this, 'add_noindex_header']);
        }

        /**
         * テキストドメインの読み込み
         */
        public function load_textdomain()
        {
            load_plugin_textdomain('gti-public-preview', false, dirname(plugin_basename(__FILE__)) . '/languages');
        }

        /**
         * 管理メニュー追加
         */
        public function add_admin_menu()
        {
            add_options_page(
                __('Public Preview Settings', 'gti-public-preview'),
                __('Public Preview', 'gti-public-preview'),
                'manage_options',
                'gti-public-preview',
                [$this, 'render_settings_page']
            );
        }

        /**
         * 設定画面描画
         */
        public function render_settings_page()
        {
            if (!current_user_can('manage_options')) return;

            // 保存処理
            if (isset($_POST['spp_save']) && check_admin_referer('spp_save_options')) {
                $key = sanitize_text_field($_POST['spp_param_key']);
                $allowed_types = isset($_POST['spp_allowed_post_types']) ? (array) $_POST['spp_allowed_post_types'] : [];
                
                update_option(self::OPTION_PARAM_KEY, $key);
                update_option(self::OPTION_ALLOWED_POST_TYPES, $allowed_types);
                
                echo '<div class="updated"><p>' . __('Settings saved.', 'gti-public-preview') . '</p></div>';
            }

            $current_key = get_option(self::OPTION_PARAM_KEY, 'public_preview');
            $current_allowed_types = get_option(self::OPTION_ALLOWED_POST_TYPES, ['post', 'page']);

            // 公開されている投稿タイプを取得
            $post_types = get_post_types(['public' => true], 'objects');
            ?>
            <div class="wrap">
                <h1><?php _e('Public Preview Settings', 'gti-public-preview'); ?></h1>
                <form method="post" action="">
                    <?php wp_nonce_field('spp_save_options'); ?>
                    <table class="form-table">
                        <tr>
                            <th scope="row"><label for="spp_param_key"><?php _e('Parameter Key', 'gti-public-preview'); ?></label></th>
                            <td>
                                <input type="text" name="spp_param_key" id="spp_param_key" value="<?php echo esc_attr($current_key); ?>" class="regular-text">
                                <p class="description"><?php _e('The URL parameter key to use. e.g. "secret_token"', 'gti-public-preview'); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><?php _e('Allowed Post Types', 'gti-public-preview'); ?></th>
                            <td>
                                <?php foreach ($post_types as $pt): ?>
                                    <label>
                                        <input type="checkbox" name="spp_allowed_post_types[]" value="<?php echo esc_attr($pt->name); ?>" <?php checked(in_array($pt->name, $current_allowed_types)); ?>>
                                        <?php echo esc_html($pt->label); ?> (<code><?php echo esc_html($pt->name); ?></code>)
                                    </label><br>
                                <?php endforeach; ?>
                                <p class="description"><?php _e('Select post types to enable public preview.', 'gti-public-preview'); ?></p>
                            </td>
                        </tr>
                    </table>
                    <p class="submit">
                        <input type="submit" name="spp_save" id="submit" class="button button-primary" value="<?php _e('Save Changes', 'gti-public-preview'); ?>">
                    </p>
                </form>
            </div>
            <?php
        }

        /**
         * メタボックス追加
         */
        public function add_custom_meta_box()
        {
            $allowed_types = get_option(self::OPTION_ALLOWED_POST_TYPES, ['post', 'page']);
            foreach ($allowed_types as $post_type) {
                add_meta_box(
                    'spp_meta_box',
                    __('Public Preview Token', 'gti-public-preview'),
                    [$this, 'render_meta_box'],
                    $post_type,
                    'side',
                    'high'
                );
            }
        }

        /**
         * メタボックスの中身描画
         */
        public function render_meta_box($post)
        {
            $token = get_post_meta($post->ID, self::META_KEY_TOKEN, true);
            $expiration = get_post_meta($post->ID, self::META_KEY_EXPIRATION, true);
            $is_active = get_post_meta($post->ID, self::META_KEY_IS_ACTIVE, true);
            $key = get_option(self::OPTION_PARAM_KEY, 'public_preview');
            
            wp_nonce_field('spp_save_meta_box_data', 'spp_meta_box_nonce');
            
            ?>
            <div style="margin-bottom: 10px;">
                <label>
                    <input type="checkbox" name="spp_is_active" value="1" <?php checked($is_active, '1'); ?>>
                    <strong><?php _e('Enable Public Preview', 'gti-public-preview'); ?></strong>
                </label>
            </div>

            <div style="margin-bottom: 10px;">
                <label for="spp_post_token"><strong><?php _e('Token:', 'gti-public-preview'); ?></strong></label><br>
                <div style="display: flex; gap: 5px;">
                    <input type="text" id="spp_post_token" name="spp_post_token" value="<?php echo esc_attr($token); ?>" style="flex: 1;">
                    <button type="button" class="button" onclick="
                        var tokenField = document.getElementById('spp_post_token');
                        tokenField.value = Math.random().toString(36).slice(-12);
                        document.querySelector('input[name=spp_is_active]').checked = true; // トークン生成時は自動で有効化
                        return false;
                    "><?php _e('Generate', 'gti-public-preview'); ?></button>
                </div>
            </div>

            <div style="margin-bottom: 10px;">
                <label for="spp_post_token_expiration"><strong><?php _e('Expiration Date (Optional):', 'gti-public-preview'); ?></strong></label><br>
                <input type="datetime-local" id="spp_post_token_expiration" name="spp_post_token_expiration" value="<?php echo esc_attr($expiration); ?>" style="width: 100%;">
                <p class="description" style="margin-top:2px; font-size:11px;">
                    <?php _e('Leave blank for no expiration.', 'gti-public-preview'); ?>
                </p>
            </div>

            <div style="margin-bottom: 15px;">
                <button type="button" class="button button-primary button-large" style="width:100%;" onclick="
                    var saveBtn = document.getElementById('save-post');
                    if(saveBtn) { 
                        saveBtn.click(); 
                    } else {
                        // Gutenberg Editor compatible
                        var publishBtn = document.querySelector('.editor-post-publish-button__button');
                        var saveDraftBtn = document.querySelector('.editor-post-save-draft'); 
                        if (saveDraftBtn) saveDraftBtn.click();
                        else if (publishBtn) publishBtn.click();
                        else document.querySelector('form#post').submit();
                    }
                ">
                    <?php _e('Save Draft / Update', 'gti-public-preview'); ?>
                </button>
            </div>
            
            <?php if ($token && $is_active): ?>
                <?php 
                    $preview_link = get_preview_post_link($post);
                    $public_link = add_query_arg($key, $token, $preview_link);
                    
                    // 期限切れチェック
                    $is_expired = false;
                    if ($expiration && strtotime($expiration) < time()) {
                        $is_expired = true;
                    }
                ?>
                <div style="margin-top: 15px; border-top: 1px solid #ddd; padding-top: 10px;">
                    <label for="spp_public_link"><strong><?php _e('Public Link:', 'gti-public-preview'); ?></strong></label><br>
                    
                    <?php if ($is_expired): ?>
                        <div style="color: #d63638; margin-bottom: 5px;">
                            <span class="dashicons dashicons-warning" style="vertical-align: text-bottom;"></span> <?php _e('Link Expired', 'gti-public-preview'); ?>
                        </div>
                    <?php endif; ?>

                    <input type="text" id="spp_public_link" value="<?php echo esc_url($public_link); ?>" style="width: 100%; margin-bottom: 5px;" readonly onclick="this.select();">
                    
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-top: 5px;">
                        <button type="button" class="button" onclick="
                            var copyText = document.getElementById('spp_public_link');
                            copyText.select();
                            copyText.setSelectionRange(0, 99999);
                            document.execCommand('copy');
                            this.innerText = '<?php echo esc_js(__('Copied!', 'gti-public-preview')); ?>';
                            setTimeout(() => { this.innerText = '<?php echo esc_js(__('Copy Link', 'gti-public-preview')); ?>'; }, 2000);
                        "><?php _e('Copy Link', 'gti-public-preview'); ?></button>

                        <a href="<?php echo esc_url($public_link); ?>" target="_blank" class="button"><?php _e('Open', 'gti-public-preview'); ?> <span class="dashicons dashicons-external" style="line-height: 1.3;"></span></a>
                    </div>
                </div>
            <?php elseif (!$is_active): ?>
                <p class="description" style="color:#d63638;"><?php _e('Public Preview is currently disabled.', 'gti-public-preview'); ?></p>
            <?php else: ?>
                <p class="description"><?php _e('Set a token and save the post to generate a public link.', 'gti-public-preview'); ?></p>
            <?php endif; ?>
            <?php
        }

        /**
         * メタデータ保存
         */
        public function save_meta_box_data($post_id)
        {
            // ノンスの確認 (POSTされていない場合は何もしない)
            if (!isset($_POST['spp_meta_box_nonce'])) return;
            if (!wp_verify_nonce($_POST['spp_meta_box_nonce'], 'spp_save_meta_box_data')) return;

            // 自動保存や権限確認
            if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;
            if (!current_user_can('edit_post', $post_id)) return;

            // トークンの保存
            if (isset($_POST['spp_post_token'])) {
                $token_val = sanitize_text_field($_POST['spp_post_token']);
                update_post_meta($post_id, self::META_KEY_TOKEN, $token_val);
            }
            
            // 有効期限の保存
            if (isset($_POST['spp_post_token_expiration'])) {
                $expiration_val = sanitize_text_field($_POST['spp_post_token_expiration']);
                update_post_meta($post_id, self::META_KEY_EXPIRATION, $expiration_val);
            }

            // 有効化フラグの保存 (チェックボックス)
            $is_active = isset($_POST['spp_is_active']) ? '1' : '0';
            update_post_meta($post_id, self::META_KEY_IS_ACTIVE, $is_active);
        }

        /**
         * プレビュー許可ロジック
         */
        public function allow_draft_preview($posts, $query)
        {
            // 管理画面、既に投稿がある場合、メインクエリ以外はスルー
            if (is_admin() || !empty($posts) || !$query->is_main_query()) {
                return $posts;
            }

            // プレビュー判定
            if (!isset($_GET['preview']) || $_GET['preview'] !== 'true') {
                return $posts;
            }

            // ID取得
            $post_id = 0;
            if (isset($_GET['p'])) {
                $post_id = intval($_GET['p']);
            } elseif (isset($_GET['page_id'])) {
                $post_id = intval($_GET['page_id']);
            }

            if ($post_id <= 0) {
                return $posts; // ID不明
            }

            // パラメータチェック
            $key = get_option(self::OPTION_PARAM_KEY, 'public_preview');
            // 投稿ごとのトークンを取得
            $token = get_post_meta($post_id, self::META_KEY_TOKEN, true);
            // 有効化フラグ
            $is_active = get_post_meta($post_id, self::META_KEY_IS_ACTIVE, true);

            // 無効化されている、または設定/トークンなし
            if ($is_active !== '1' || empty($key) || empty($token)) {
                return $posts; 
            }

            if (!isset($_GET[$key]) || $_GET[$key] !== $token) {
                return $posts; // パラメータ不一致
            }

            // 有効期限チェック
            $expiration = get_post_meta($post_id, self::META_KEY_EXPIRATION, true);
            if ($expiration && strtotime($expiration) < time()) {
                return $posts; // 期限切れ
            }

            // ここまで来たら許可対象
            $post = get_post($post_id);
            // 投稿が存在し、かつ公開前のステータスであること
            if ($post && in_array($post->post_status, ['draft', 'pending', 'future'])) {
                // 許可された投稿タイプかチェック
                $allowed_types = get_option(self::OPTION_ALLOWED_POST_TYPES, ['post', 'page']);
                if (!in_array($post->post_type, $allowed_types)) {
                    return $posts;
                }

                // 投稿セット
                $posts = [$post];
                // 404回避
                $query->is_404 = false;
                if ($post->post_type === 'page') {
                    $query->is_page = true;
                } else {
                    $query->is_single = true;
                }
            }

            return $posts;
        }

        /**
         * プレビューリンクにパラメータを付与
         */
        public function filter_preview_link($link, $post) {
            $key = get_option(self::OPTION_PARAM_KEY, 'public_preview');
            $token = get_post_meta($post->ID, self::META_KEY_TOKEN, true);
            // 有効化フラグ
            $is_active = get_post_meta($post->ID, self::META_KEY_IS_ACTIVE, true);
            
            if ($key && $token && ($is_active === '1') && in_array($post->post_status, ['draft', 'pending', 'future'])) {
                // 許可された投稿タイプかチェック
                $allowed_types = get_option(self::OPTION_ALLOWED_POST_TYPES, ['post', 'page']);
                if (in_array($post->post_type, $allowed_types)) {
                    return add_query_arg($key, $token, $link);
                }
            }
            return $link;
        }

        /**
         * 管理バーに「公開プレビューリンク」を表示（投稿編集画面など）
         */
        public function add_preview_link_to_admin_bar($wp_admin_bar)
        {
            if (!is_admin()) return;

            $screen = get_current_screen();
            // 投稿編集画面のみ
            if (!$screen || $screen->base !== 'post') return;

            global $post;
            if (!$post || !in_array($post->post_status, ['draft', 'pending', 'future'])) return;

            $allowed_types = get_option(self::OPTION_ALLOWED_POST_TYPES, ['post', 'page']);
            if (!in_array($post->post_type, $allowed_types)) return;

            $key = get_option(self::OPTION_PARAM_KEY, 'public_preview');
            $token = get_post_meta($post->ID, self::META_KEY_TOKEN, true);
            $is_active = get_post_meta($post->ID, self::META_KEY_IS_ACTIVE, true);
            
            if (empty($key) || empty($token) || $is_active !== '1') return;

            // 有効期限チェック
            $expiration = get_post_meta($post->ID, self::META_KEY_EXPIRATION, true);
            if ($expiration && strtotime($expiration) < time()) {
                return; // 期限切れの場合は表示しない
            }

            // プレビューリンク生成
            $preview_link = get_preview_post_link($post);
            $public_link = add_query_arg($key, $token, $preview_link);

            $wp_admin_bar->add_node([
                'id'    => 'spp-public-preview-link',
                'parent' => 'top-secondary', // 右側に配置
                'title' => '<span class="dashicons dashicons-visibility" style="margin-top:4px;"></span> ' . __('Public Preview Link', 'gti-public-preview'),
                'href'  => $public_link,
                'meta'  => [
                    'target' => '_blank',
                    'title'  => __('Open Public Preview in new tab', 'gti-public-preview'),
                ],
            ]);
        }

        /**
         * noindex メタタグを出力
         */
        public function add_noindex_meta()
        {
            if ($this->is_public_preview()) {
                echo '<meta name="robots" content="noindex,nofollow" />' . "\n";
            }
        }

        /**
         * noindex ヘッダーを追加
         */
        public function add_noindex_header($headers)
        {
            if ($this->is_public_preview()) {
                $headers['X-Robots-Tag'] = 'noindex, nofollow';
            }
            return $headers;
        }

        /**
         * 現在のリクエストが Public Preview かどうか判定
         */
        private function is_public_preview()
        {
            if (is_admin()) return false;
            
            // プレビューパラメータがあるか
            if (!isset($_GET['preview']) || $_GET['preview'] !== 'true') return false;

            // トークンパラメータがあるか
            $key = get_option(self::OPTION_PARAM_KEY, 'public_preview');
            if (empty($key) || !isset($_GET[$key])) return false;

            // ※厳密なトークンチェックまではここでしなくても、
            // allow_draft_preview を通過していれば表示されているはずなので
            // 表示されている＝プレビュー中とみなすこともできるが、念のため
            
            return true;
        }
    }

    /**
     * 自動アップデート用クラス
     */
    if (!class_exists('GTI_Public_Preview_Updater')) {
        class GTI_Public_Preview_Updater
        {
            private $slug;
            private $plugin_file;
            private $update_url;
            private $cache_key;
            private $cache_ttl = 43200; // 12時間

            public function __construct($plugin_file, $update_url)
            {
                $this->plugin_file = $plugin_file;
                $this->slug = plugin_basename($plugin_file);
                $this->update_url = $update_url;
                $this->cache_key = 'gti_updater_' . md5($this->slug);

                add_filter('pre_set_site_transient_update_plugins', [$this, 'check_update']);
                add_filter('plugins_api', [$this, 'plugin_info'], 10, 3);
            }

            public function check_update($transient)
            {
                if (empty($transient->checked)) {
                    return $transient;
                }

                $remote = $this->request_info();

                if (
                    $remote && 
                    version_compare($remote->version, $transient->checked[$this->slug], '>')
                ) {
                    $obj = new stdClass();
                    $obj->slug = $this->slug;
                    $obj->new_version = $remote->version;
                    $obj->url = $remote->url;
                    $obj->package = $remote->download_url;
                    
                    // アイコンなどがあれば
                    if (!empty($remote->icons)) $obj->icons = (array)$remote->icons;
                    if (!empty($remote->banners)) $obj->banners = (array)$remote->banners;

                    $transient->response[$this->slug] = $obj;
                }

                return $transient;
            }

            public function plugin_info($res, $action, $args)
            {
                if ($action !== 'plugin_information') return $res;
                if ($args->slug !== $this->slug && $args->slug !== dirname($this->slug)) return $res;

                $remote = $this->request_info();
                if (!$remote) return $res;

                $res = new stdClass();
                $res->name = $remote->name;
                $res->slug = $this->slug; // full path slug
                $res->version = $remote->version;
                $res->tested = $remote->tested;
                $res->requires = $remote->requires;
                $res->author = $remote->author;
                $res->author_profile = $remote->author_profile ?? '';
                $res->download_link = $remote->download_url;
                $res->trunk = $remote->download_url;
                $res->requires_php = $remote->requires_php;
                $res->last_updated = $remote->last_updated;
                $res->sections = (array) $remote->sections;
                
                if (!empty($remote->icons)) $res->icons = (array)$remote->icons;
                if (!empty($remote->banners)) $res->banners = (array)$remote->banners;

                return $res;
            }

            private function request_info()
            {
                // キャッシュ確認
                $cache = get_transient($this->cache_key);
                if ($cache) return $cache;

                if (empty($this->update_url)) return false;

                $response = wp_remote_get($this->update_url, [
                    'timeout' => 10,
                    'headers' => ['Accept' => 'application/json']
                ]);

                if (is_wp_error($response) || wp_remote_retrieve_response_code($response) !== 200) {
                    return false;
                }

                $body = wp_remote_retrieve_body($response);
                $data = json_decode($body);

                if (json_last_error() !== JSON_ERROR_NONE || empty($data)) {
                    return false;
                }

                // キャッシュセット
                set_transient($this->cache_key, $data, $this->cache_ttl);

                return $data;
            }
        }
    }

    $gti_public_preview = new GTI_Public_Preview();
    
    // config.cgi からアップデート用URLを読み込む
    $config_file = dirname(__FILE__) . '/config.cgi';
    $update_url = '';
    
    if (file_exists($config_file)) {
        // 先頭の1行を取得してトリム
        $lines = file($config_file, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        if (!empty($lines)) {
            $update_url = trim($lines[0]);
        }
    }

    // URLが設定されている場合のみアップデーターを有効化
    if (!empty($update_url)) {
        new GTI_Public_Preview_Updater(
            __FILE__, 
            $update_url
        );
    }
}
