<?php
/**
 * GTI IP Gate - Admin Dashboard with Authentication
 */
session_start();

$conf_file = __DIR__ . '/allow_ips_conf.cgi';
$auth_file = __DIR__ . '/admin_conf.cgi';

// Ensure config files exist
if (!file_exists($conf_file)) {
    if (file_exists(__DIR__ . '/allow_ips_conf.sample.cgi')) {
        copy(__DIR__ . '/allow_ips_conf.sample.cgi', $conf_file);
    } else {
        file_put_contents($conf_file, "# Allowed IPs\n");
    }
}

if (!file_exists($auth_file)) {
    if (file_exists(__DIR__ . '/admin_conf.sample.cgi')) {
        copy(__DIR__ . '/admin_conf.sample.cgi', $auth_file);
    } else {
        file_put_contents($auth_file, "admin:admin");
    }
}

// Read Auth Credentials
function get_auth_credentials($file) {
    if (!file_exists($file)) return ['admin', 'admin'];
    $content = trim(file_get_contents($file));
    if (strpos($content, ':') !== false) {
        return explode(':', $content, 2);
    }
    return ['admin', 'admin'];
}

// Save Auth Credentials
function save_auth_credentials($file, $user, $pass) {
    return file_put_contents($file, "$user:$pass") !== false;
}

[$admin_user, $admin_pass] = get_auth_credentials($auth_file);

// Handle Logout
if (isset($_GET['logout'])) {
    session_destroy();
    header('Location: ' . strtok($_SERVER['REQUEST_URI'], '?'));
    exit;
}

// Handle Login
$login_error = '';
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['login'])) {
    $user = $_POST['username'] ?? '';
    $pass = $_POST['password'] ?? '';
    
    if ($user === $admin_user && $pass === $admin_pass) {
        $_SESSION['authenticated'] = true;
        header('Location: ' . $_SERVER['REQUEST_URI']);
        exit;
    } else {
        $login_error = 'IDまたはパスワードが正しくありません。';
    }
}

// Check Authentication
$is_authenticated = isset($_SESSION['authenticated']) && $_SESSION['authenticated'] === true;

// API Response handler (Protected)
if (isset($_GET['api'])) {
    header('Content-Type: application/json');
    
    if (!$is_authenticated) {
        http_response_code(401);
        echo json_encode(['success' => false, 'message' => 'Unauthorized']);
        exit;
    }

    $ips = read_ips($conf_file);
    
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        $data = json_decode(file_get_contents('php://input'), true);
        $action = $data['action'] ?? '';
        $success = false;
        $message = '';
        
        if ($action === 'add') {
            $new_ip = trim($data['ip'] ?? '');
            if ($new_ip) {
                if (!in_array($new_ip, $ips)) {
                    $ips[] = $new_ip;
                    if (save_ips($conf_file, $ips)) {
                        $success = true;
                        $message = '加されました：' . $new_ip;
                    } else {
                        $message = '保存に失敗しました。';
                    }
                } else {
                    $message = '既に登録されています。';
                }
            } else {
                $message = '無効なIPアドレスです。';
            }
        } elseif ($action === 'delete') {
            $delete_ip = $data['ip'] ?? '';
            $new_list = array_filter($ips, fn($ip) => $ip !== $delete_ip);
            if (count($new_list) !== count($ips)) {
                if (save_ips($conf_file, $new_list)) {
                    $ips = array_values($new_list);
                    $success = true;
                    $message = '削除されました：' . $delete_ip;
                } else {
                    $message = '保存に失敗しました。';
                }
            } else {
                $message = 'IPが見つかりませんでした。';
            }
        } elseif ($action === 'update_auth') {
            $new_u = trim($data['username'] ?? '');
            $new_p = trim($data['password'] ?? '');
            if ($new_u && $new_p) {
                if (save_auth_credentials($auth_file, $new_u, $new_p)) {
                    $success = true;
                    $message = '認証情報を更新しました。';
                } else {
                    $message = '保存に失敗しました。';
                }
            } else {
                $message = 'IDとパスワードを入力してください。';
            }
        }
        
        echo json_encode(['success' => $success, 'message' => $message, 'ips' => $ips]);
        exit;
    }
    
    echo json_encode(['ips' => $ips]);
    exit;
}

function read_ips($file) {
    if (!file_exists($file)) return [];
    $lines = file($file, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    $ips = [];
    foreach ($lines as $line) {
        $line = trim($line);
        if ($line === '' || strpos($line, '#') === 0) continue;
        $ips[] = $line;
    }
    return array_values(array_unique($ips));
}

function save_ips($file, $ips) {
    $content = "# Allowed IPs\n# Generated by GTI IP Gate Admin\n";
    foreach ($ips as $ip) {
        $ip = trim($ip);
        if ($ip !== '') {
            $content .= $ip . "\n";
        }
    }
    return file_put_contents($file, $content) !== false;
}

$current_ip = $_SERVER['REMOTE_ADDR'] ?? '';
$ips = read_ips($conf_file);
?>
<!DOCTYPE html>
<html lang="ja">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>GTI IP Gate 管理パネル</title>
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Outfit:wght@300;400;600&family=Inter:wght@300;400;600&display=swap" rel="stylesheet">
    <style>
        :root {
            --primary: #6366f1;
            --primary-hover: #4f46e5;
            --bg: #0f172a;
            --card-bg: rgba(30, 41, 59, 0.7);
            --text: #f8fafc;
            --text-dim: #94a3b8;
            --danger: #ef4444;
            --success: #10b981;
            --border: rgba(255, 255, 255, 0.1);
        }

        * {
            box-sizing: border-box;
            margin: 0;
            padding: 0;
        }

        body {
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif;
            background-color: var(--bg);
            background-image: 
                radial-gradient(circle at 0% 0%, rgba(99, 102, 241, 0.15) 0%, transparent 50%),
                radial-gradient(circle at 100% 100%, rgba(139, 92, 246, 0.15) 0%, transparent 50%);
            color: var(--text);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }

        .container {
            width: 100%;
            max-width: 600px;
            animation: fadeIn 0.6s ease-out;
        }

        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(20px); }
            to { opacity: 1; transform: translateY(0); }
        }

        header {
            margin-bottom: 32px;
            text-align: center;
        }

        h1 {
            font-family: 'Outfit', sans-serif;
            font-size: 2.5rem;
            font-weight: 600;
            margin-bottom: 8px;
            background: linear-gradient(135deg, #fff 0%, #94a3b8 100%);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
        }

        .status-badge {
            display: inline-block;
            padding: 4px 12px;
            border-radius: 9999px;
            background: rgba(16, 185, 129, 0.1);
            color: var(--success);
            font-size: 0.75rem;
            font-weight: 600;
            border: 1px solid rgba(16, 185, 129, 0.2);
        }

        .card {
            background: var(--card-bg);
            backdrop-filter: blur(12px);
            -webkit-backdrop-filter: blur(12px);
            border: 1px solid var(--border);
            border-radius: 24px;
            padding: 32px;
            box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.5);
        }

        .login-title {
            text-align: center;
            font-size: 1.5rem;
            margin-bottom: 24px;
            color: var(--text);
            font-family: 'Outfit', sans-serif;
        }

        .login-form {
            display: flex;
            flex-direction: column;
            gap: 16px;
        }

        .error-msg {
            background: rgba(239, 68, 68, 0.1);
            color: var(--danger);
            padding: 12px;
            border-radius: 12px;
            font-size: 0.875rem;
            border: 1px solid rgba(239, 68, 68, 0.2);
            margin-bottom: 16px;
        }

        .current-ip-info {
            background: rgba(255, 255, 255, 0.05);
            border-radius: 16px;
            padding: 16px;
            margin-bottom: 24px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            border: 1px dashed rgba(255, 255, 255, 0.1);
        }

        .current-ip-label {
            font-size: 0.875rem;
            color: var(--text-dim);
        }

        .current-ip-value {
            font-family: 'Courier New', Courier, monospace;
            font-size: 1.1rem;
            color: var(--primary);
            font-weight: 600;
        }

        .add-form {
            display: flex;
            gap: 12px;
            margin-bottom: 32px;
        }

        input[type="text"], input[type="password"] {
            flex: 1;
            background: rgba(15, 23, 42, 0.5);
            border: 1px solid var(--border);
            border-radius: 12px;
            padding: 14px 16px;
            color: white;
            font-size: 1rem;
            transition: all 0.3s ease;
        }

        input:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 4px rgba(99, 102, 241, 0.2);
        }

        .btn {
            cursor: pointer;
            border: none;
            border-radius: 12px;
            padding: 12px 24px;
            font-weight: 600;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 8px;
            font-size: 0.95rem;
        }

        .btn-primary {
            background: var(--primary);
            color: white;
        }

        .btn-primary:hover {
            background: var(--primary-hover);
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(99, 102, 241, 0.3);
        }

        .btn-ghost {
            background: transparent;
            color: var(--text-dim);
            padding: 8px 16px;
            font-size: 0.875rem;
            border: 1px solid transparent;
        }

        .btn-ghost:hover {
            color: white;
            background: rgba(255, 255, 255, 0.05);
            border-color: var(--border);
        }

        .ip-list-title {
            font-size: 0.875rem;
            color: var(--text-dim);
            margin-bottom: 12px;
            padding-left: 4px;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .ip-list {
            list-style: none;
        }

        .ip-item {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 12px 16px;
            border-radius: 12px;
            background: rgba(255, 255, 255, 0.03);
            margin-bottom: 8px;
            transition: all 0.3s ease;
            border: 1px solid transparent;
        }

        .ip-item:hover {
            background: rgba(255, 255, 255, 0.05);
            border-color: var(--border);
        }

        .ip-address {
            font-family: 'Courier New', Courier, monospace;
            font-size: 1rem;
            letter-spacing: 0.5px;
        }

        .delete-btn {
            color: var(--danger);
            opacity: 0.5;
            background: none;
            border: none;
            cursor: pointer;
            padding: 8px;
            border-radius: 8px;
            transition: all 0.2s;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        .delete-btn:hover {
            opacity: 1;
            background: rgba(239, 68, 68, 0.1);
            transform: scale(1.05);
        }

        .empty-state {
            text-align: center;
            color: var(--text-dim);
            padding: 40px 0;
            font-style: italic;
            background: rgba(255, 255, 255, 0.02);
            border-radius: 12px;
            border: 1px dashed var(--border);
        }

        .logout-link {
            color: var(--text-dim);
            text-decoration: none;
            font-size: 0.75rem;
        }
        .logout-link:hover {
            color: var(--danger);
        }

        .settings-toggle {
            cursor: pointer;
            font-size: 0.75rem;
            color: var(--text-dim);
        }
        .settings-toggle:hover { color: white; }

        .settings-panel {
            display: none;
            padding-top: 24px;
            border-top: 1px solid var(--border);
            margin-top: 24px;
            animation: slideDown 0.3s ease-out;
        }
        @keyframes slideDown {
            from { opacity: 0; transform: translateY(-10px); }
            to { opacity: 1; transform: translateY(0); }
        }

        footer {
            margin-top: 32px;
            text-align: center;
            color: var(--text-dim);
            font-size: 0.875rem;
        }

        footer a {
            color: var(--primary);
            text-decoration: none;
            font-weight: 600;
        }

        #toast-container {
            position: fixed;
            bottom: 24px;
            right: 24px;
            z-index: 1000;
        }

        .toast {
            background: #1e293b;
            color: white;
            padding: 16px 24px;
            border-radius: 12px;
            box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.3);
            margin-top: 12px;
            border-left: 4px solid var(--primary);
            display: flex;
            align-items: center;
            gap: 12px;
            min-width: 250px;
            animation: slideIn 0.3s cubic-bezier(0.16, 1, 0.3, 1);
        }

        @keyframes slideIn {
            from { transform: translateX(100%); opacity: 0; }
            to { transform: translateX(0); opacity: 1; }
        }
    </style>
</head>
<body>

<div class="container">
    <header>
        <h1>GTI IP Gate</h1>
        <span class="status-badge">アクセス制限稼働中</span>
    </header>

    <?php if (!$is_authenticated): ?>
    <div class="card">
        <h2 class="login-title">認証が必要です</h2>
        
        <div class="current-ip-info" style="margin-bottom: 24px;">
            <div style="width: 100%; text-align: center;">
                <div class="current-ip-label">あなたの現在のIPアドレス</div>
                <div class="current-ip-value"><?php echo htmlspecialchars($current_ip); ?></div>
            </div>
        </div>

        <?php if ($login_error): ?>
            <div class="error-msg"><?php echo htmlspecialchars($login_error); ?></div>
        <?php endif; ?>
        <form method="POST" class="login-form">
            <input type="text" name="username" placeholder="ログインID" required autofocus>
            <input type="password" name="password" placeholder="パスワード" required>
            <button type="submit" name="login" class="btn btn-primary">ログイン</button>
        </form>
    </div>
    <?php else: ?>
    <div class="card">
        <div class="current-ip-info">
            <div>
                <div class="current-ip-label">あなたの現在のIPアドレス</div>
                <div class="current-ip-value" id="my-ip"><?php echo htmlspecialchars($current_ip); ?></div>
            </div>
            <button class="btn btn-ghost" onclick="useMyIp()">このIPを追加</button>
        </div>

        <form class="add-form" id="add-form">
            <input type="text" id="ip-input" placeholder="IPまたはCIDR (例: 1.2.3.4/24)" required autocomplete="off">
            <button type="submit" class="btn btn-primary">追加</button>
        </form>

        <div class="ip-list-title">
            <span>許可済みIP一覧</span>
            <a href="?logout=1" class="logout-link">ログアウト</a>
        </div>
        
        <div id="ip-list-container">
            <ul class="ip-list" id="ip-list">
                <?php foreach ($ips as $ip): ?>
                <li class="ip-item" data-ip="<?php echo htmlspecialchars($ip); ?>">
                    <span class="ip-address"><?php echo htmlspecialchars($ip); ?></span>
                    <button class="delete-btn" onclick="deleteIp('<?php echo htmlspecialchars($ip); ?>')" title="削除">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M3 6h18"></path><path d="M19 6v14c0 1-1 2-2 2H7c-1 0-2-1-2-2V6"></path><path d="M8 6V4c0-1 1-2 2-2h4c1 0 2 1 2 2v2"></path><line x1="10" y1="11" x2="10" y2="17"></line><line x1="14" y1="11" x2="14" y2="17"></line></svg>
                    </button>
                </li>
                <?php endforeach; ?>
                <?php if (empty($ips)): ?>
                <li class="empty-state">許可されたIPアドレスはありません</li>
                <?php endif; ?>
            </ul>
        </div>

        <div style="text-align: right; margin-top: 24px;">
            <span class="settings-toggle" onclick="toggleSettings()">⚙️ ログイン情報の変更</span>
        </div>

        <div id="settings-panel" class="settings-panel">
            <div class="login-form">
                <input type="text" id="new-username" placeholder="新しいログインID" value="<?php echo htmlspecialchars($admin_user); ?>">
                <input type="text" id="new-password" placeholder="新しいパスワード" value="<?php echo htmlspecialchars($admin_pass); ?>">
                <button class="btn btn-ghost" onclick="updateAuth()">更新する</button>
            </div>
        </div>
    </div>
    <?php endif; ?>

    <footer>
        <p>&copy; <?php echo date('Y'); ?> <a href="https://gti.co.jp" target="_blank">GTI</a> | IPベース・アクセスゲート</p>
    </footer>
</div>

<div id="toast-container"></div>

<script>
    <?php if ($is_authenticated): ?>
    const ipList = document.getElementById('ip-list');
    const ipInput = document.getElementById('ip-input');
    const addForm = document.getElementById('add-form');
    const myIp = document.getElementById('my-ip').textContent;

    function showToast(message, type = 'primary') {
        const toast = document.createElement('div');
        toast.className = 'toast';
        toast.textContent = message;
        if (type === 'danger') toast.style.borderLeftColor = 'var(--danger)';
        if (type === 'success') toast.style.borderLeftColor = 'var(--success)';
        
        document.getElementById('toast-container').appendChild(toast);
        setTimeout(() => {
            toast.style.opacity = '0';
            toast.style.transform = 'translateX(20px)';
            toast.style.transition = 'all 0.5s ease';
            setTimeout(() => toast.remove(), 500);
        }, 3000);
    }

    async function fetchData(action, payload = {}) {
        try {
            const response = await fetch('?api=1', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ action, ...payload })
            });
            const data = await response.json();
            if (data.success) {
                if (data.ips) renderList(data.ips);
                showToast(data.message, 'success');
            } else if (data.message) {
                showToast(data.message, 'danger');
            }
        } catch (err) {
            showToast('通信エラーが発生しました', 'danger');
        }
    }

    function renderList(ips) {
        if (ips.length === 0) {
            ipList.innerHTML = '<li class="empty-state">許可されたIPアドレスはありません</li>';
            return;
        }
        ipList.innerHTML = ips.map(ip => `
            <li class="ip-item" data-ip="${ip}">
                <span class="ip-address">${ip}</span>
                <button class="delete-btn" onclick="deleteIp('${ip}')" title="削除">
                    <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M3 6h18"></path><path d="M19 6v14c0 1-1 2-2 2H7c-1 0-2-1-2-2V6"></path><path d="M8 6V4c0-1 1-2 2-2h4c1 0 2 1 2 2v2"></path><line x1="10" y1="11" x2="10" y2="17"></line><line x1="14" y1="11" x2="14" y2="17"></line></svg>
                </button>
            </li>
        `).join('');
    }

    addForm.addEventListener('submit', (e) => {
        e.preventDefault();
        const ip = ipInput.value.trim();
        if (ip) {
            fetchData('add', { ip });
            ipInput.value = '';
        }
    });

    function deleteIp(ip) {
        if (confirm(`IPアドレス「${ip}」を許可リストから削除しますか？`)) {
            fetchData('delete', { ip });
        }
    }

    function useMyIp() {
        ipInput.value = myIp;
        ipInput.focus();
    }

    function toggleSettings() {
        const panel = document.getElementById('settings-panel');
        panel.style.display = panel.style.display === 'block' ? 'none' : 'block';
    }

    function updateAuth() {
        const username = document.getElementById('new-username').value.trim();
        const password = document.getElementById('new-password').value.trim();
        if (username && password) {
            if (confirm('ログイン情報を更新します。よろしいですか？')) {
                fetchData('update_auth', { username, password });
            }
        }
    }
    <?php endif; ?>
</script>

</body>
</html>
