<?php
/**
 * Handle Form Submission
 */

function gti_forms_handle_submit() {
	check_ajax_referer( 'gti_forms_action', 'gti_forms_nonce' );

	$form_id = isset( $_POST['form_id'] ) ? intval( $_POST['form_id'] ) : 0;
	if ( ! $form_id ) {
		wp_send_json_error( array( 'message' => __( '無効なフォームIDです。', 'gti-forms' ) ) );
	}

	// Get form fields to map names to labels
	$post = get_post( $form_id );
	$blocks = parse_blocks( $post->post_content );
	$field_map = array();
	foreach ( $blocks as $block ) {
		if ( isset( $block['attrs']['name'] ) && isset( $block['attrs']['label'] ) ) {
			$field_map[ $block['attrs']['name'] ] = $block['attrs']['label'];
		}
	}

	$submitted_data = array();
	$raw_data = array();
	$email_to_customer = '';
	$stripe_block = null;
	
	foreach ( $blocks as $block ) {
		if ( $block['blockName'] === 'gti/form-stripe' ) {
			$stripe_block = $block;
		}
	}

	// Get field types to determine sanitization
	$field_types = array();
	foreach ( $blocks as $block ) {
		if ( isset( $block['attrs']['name'] ) ) {
			$field_types[ $block['attrs']['name'] ] = $block['blockName'];
		}
	}

	foreach ( $_POST as $key => $value ) {
		// Ignore internal WP fields and our own system fields
		if ( 
			in_array( $key, array( 'action', 'form_id', 'gti_forms_nonce', 'mode' ) ) || 
			strpos( $key, '_wp' ) === 0 
		) {
			continue;
		}
		
		// Map labels for display
		if ( isset( $field_map[ $key ] ) ) {
			$label = $field_map[ $key ];
			$block_type = isset( $field_types[ $key ] ) ? $field_types[ $key ] : '';

			// Use appropriate sanitization based on field type
			if ( $block_type === 'gti/form-textarea' ) {
				$clean_value = sanitize_textarea_field( $value );
			} else {
				$clean_value = sanitize_text_field( $value );
			}

			$submitted_data[ $label ] = $clean_value;
			$raw_data[ $key ] = $clean_value;
		}
	}

	/**
	 * Filter to allow adding/modifying data based on context (GET/POST params)
	 * Useful for reservation systems (e.g., adding Plan Name or Menu from URL)
	 */
	$submitted_data = apply_filters( 'gti_forms_submitted_data', $submitted_data, $form_id );

	// Detect email for customer notification (after filter in case email was injected)
	foreach ( $submitted_data as $label => $value ) {
		if ( strpos( strtolower($label), 'email' ) !== false || strpos( strtolower($label), 'メール' ) !== false || is_email( $value ) ) {
			$email_to_customer = sanitize_email( $value );
		}
	}

	// 1. Confirmation Mode
	if ( isset($_POST['mode']) && $_POST['mode'] === 'confirm' ) {
		$html = '<div class="gti-confirm-table">';
		foreach ( $submitted_data as $label => $value ) {
			$html .= '<div class="gti-confirm-row">';
			$html .= '<span class="gti-confirm-label">' . esc_html($label) . '</span>';
			$html .= '<span class="gti-confirm-value">' . nl2br(esc_html($value)) . '</span>';
			$html .= '</div>';
		}
		$html .= '</div>';

		$button_text = __( '送信する', 'gti-forms' );
		foreach($blocks as $block) {
			if ($block['blockName'] === 'gti/form-submit' && isset($block['attrs']['text'])) {
				$button_text = $block['attrs']['text'];
			} elseif ($block['blockName'] === 'gti/form-stripe' && isset($block['attrs']['label'])) {
				$button_text = $block['attrs']['label'];
			}
		}

		wp_send_json_success( array(
			'screen' => 'confirmation',
			'html' => $html,
			'button_text' => $button_text
		) );
	}

	// 2. Final Submission Mode
	// Validation Hook (Allow third-party plugins like Throws SPAM Away to validate)
	$validation_error = apply_filters( 'gti_forms_validate', null, $form_id, $submitted_data );
	if ( is_wp_error( $validation_error ) ) {
		wp_send_json_error( array( 'message' => $validation_error->get_error_message() ) );
	}

	// Trigger before submit hook
	do_action( 'gti_forms_before_submit', $form_id, $submitted_data );

	// If Stripe is present, create session instead of final completion (for simplicity, we'll save the inquiry as 'pending' or just save it later)
	if ( $stripe_block ) {
		$checkout_session = gti_forms_create_stripe_session( $stripe_block, $form_id, $submitted_data );
		if ( $checkout_session ) {
			wp_send_json_success( array(
				'requires_payment' => true,
				'checkout_session_id' => $checkout_session->id
			) );
		} else {
			wp_send_json_error( array( 'message' => __( '決済セッションの作成に失敗しました。', 'gti-forms' ) ) );
		}
	}

	// Determine inquiry post title (allow dynamic override)
	$inquiry_title = sprintf( __( 'お問い合わせ: %s (%s)', 'gti-forms' ), ( isset( $submitted_data[__( 'お名前', 'gti-forms' )] ) ? $submitted_data[__( 'お名前', 'gti-forms' )] : __( '不明', 'gti-forms' ) ), current_time('Y-m-d H:i') );
	$inquiry_title = apply_filters( 'gti_forms_inquiry_post_title', $inquiry_title, $form_id, $submitted_data );

	// Create inquiry post (Standard submission)
	$inquiry_id = wp_insert_post( array(
		'post_type'    => 'gti-inquiry',
		'post_title'   => $inquiry_title,
		'post_status'  => 'publish',
		'post_content' => wp_json_encode( $submitted_data, JSON_UNESCAPED_UNICODE ),
	) );

	if ( is_wp_error( $inquiry_id ) ) {
		wp_send_json_error( array( 'message' => __( '保存に失敗しました。', 'gti-forms' ) ) );
	}

	// Save original data and form ID
	update_post_meta( $inquiry_id, '_gti_data', $submitted_data );
	update_post_meta( $inquiry_id, '_gti_form_id', $form_id );

	// Send Emails
	gti_forms_send_emails( $form_id, $submitted_data, $email_to_customer, $raw_data );

	// Trigger after submit hook
	do_action( 'gti_forms_after_submit', $inquiry_id, $form_id, $submitted_data );

	// Determine redirect URL (allow dynamic override)
	$redirect_url = add_query_arg('gti_step', 'thanks', wp_get_referer());
	$redirect_url = apply_filters( 'gti_forms_redirect_url', $redirect_url, $form_id, $submitted_data );

	wp_send_json_success( array( 
		'message' => __( 'お問い合わせを送信しました。', 'gti-forms' ),
		'redirect_url' => $redirect_url 
	) );
}
add_action( 'wp_ajax_gti_forms_submit', 'gti_forms_handle_submit' );
add_action( 'wp_ajax_nopriv_gti_forms_submit', 'gti_forms_handle_submit' );

/**
 * Stripe Checkout Session Creation
 */
function gti_forms_create_stripe_session( $block, $form_id, $data ) {
	$secret_key = get_option('gti_forms_stripe_secret_key');
	if ( ! $secret_key ) return false;

	\Stripe\Stripe::setApiKey($secret_key);

	// In a real plugin, we would use composer or a bundled copy. 
	// For this demo, let's assume we use the global Stripe object if available or just mockup the logic.
	// NOTE: For a real implementation, you'd need the Stripe library files.
	
	try {
		$session_args = [
			'payment_method_types' => ['card'],
			'line_items' => [[
				'price_data' => [
					'currency' => 'jpy',
					'product_data' => [
						'name' => get_the_title($form_id),
					],
					'unit_amount' => $block['attrs']['amount'],
				],
				'quantity' => 1,
			]],
			'mode' => 'payment',
			'success_url' => add_query_arg('gti_form_status', 'success', get_permalink()),
			'cancel_url' => add_query_arg('gti_form_status', 'cancel', get_permalink()),
			'metadata' => [
				'form_id' => $form_id,
				'data' => wp_json_encode($data, JSON_UNESCAPED_UNICODE)
			]
		];

		// Filter to allow dynamic amount calculation (Cart functionality)
		$session_args = apply_filters( 'gti_forms_stripe_session_args', $session_args, $form_id, $data );

		$session = \Stripe\Checkout\Session::create($session_args);
		return $session;
	} catch (Exception $e) {
		return false;
	}
}
